<?php
class MySQLSchemeRepository implements SchemeRepositoryInterface {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    public function save(SchemeOfWork $scheme) {
        $stmt = $this->db->prepare("
            INSERT INTO schemes (title, description, subject, grade_level, term, year, file_path, file_type, file_size, is_active, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $scheme->getTitle(),
            $scheme->getDescription(),
            $scheme->getSubject(),
            $scheme->getGradeLevel(),
            $scheme->getTerm(),
            $scheme->getYear(),
            $scheme->getFilePath(),
            $scheme->getFileType(),
            $scheme->getFileSize(),
            $scheme->isActive() ? 1 : 0
        ]);
        
        $scheme->setId($this->db->lastInsertId());
        return $scheme;
    }
    
    public function findById($id) {
        $stmt = $this->db->prepare("SELECT * FROM schemes WHERE id = ?");
        $stmt->execute([$id]);
        $data = $stmt->fetch();
        
        if (!$data) return null;
        
        return $this->hydrate($data);
    }
    
    public function findAll($filters = []) {
        $sql = "SELECT * FROM schemes WHERE is_active = 1";
        $params = [];
        
        if (!empty($filters['subject'])) {
            $sql .= " AND subject = ?";
            $params[] = $filters['subject'];
        }
        
        if (!empty($filters['grade_level'])) {
            $sql .= " AND grade_level = ?";
            $params[] = $filters['grade_level'];
        }
        
        if (!empty($filters['term'])) {
            $sql .= " AND term = ?";
            $params[] = $filters['term'];
        }
        
        if (!empty($filters['year'])) {
            $sql .= " AND year = ?";
            $params[] = $filters['year'];
        }
        
        $sql .= " ORDER BY subject, grade_level, term";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll();
        
        $schemes = [];
        foreach ($results as $data) {
            $schemes[] = $this->hydrate($data);
        }
        
        return $schemes;
    }
    
    public function findBySubject($subject) {
        $stmt = $this->db->prepare("
            SELECT * FROM schemes 
            WHERE subject = ? AND is_active = 1 
            ORDER BY grade_level, term
        ");
        $stmt->execute([$subject]);
        $results = $stmt->fetchAll();
        
        $schemes = [];
        foreach ($results as $data) {
            $schemes[] = $this->hydrate($data);
        }
        
        return $schemes;
    }
    
    public function findByGradeLevel($gradeLevel) {
        $stmt = $this->db->prepare("
            SELECT * FROM schemes 
            WHERE grade_level = ? AND is_active = 1 
            ORDER BY subject, term
        ");
        $stmt->execute([$gradeLevel]);
        $results = $stmt->fetchAll();
        
        $schemes = [];
        foreach ($results as $data) {
            $schemes[] = $this->hydrate($data);
        }
        
        return $schemes;
    }   public function update(SchemeOfWork $scheme) {
        $stmt = $this->db->prepare("
            UPDATE schemes 
            SET title = ?, description = ?, subject = ?, grade_level = ?, term = ?, year = ?, 
                file_path = ?, file_type = ?, file_size = ?, is_active = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([
            $scheme->getTitle(),
            $scheme->getDescription(),
            $scheme->getSubject(),
            $scheme->getGradeLevel(),
            $scheme->getTerm(),
            $scheme->getYear(),
            $scheme->getFilePath(),
            $scheme->getFileType(),
            $scheme->getFileSize(),
            $scheme->isActive() ? 1 : 0,
            $scheme->getId()
        ]);
        
        return $scheme;
    }
    
    private function hydrate($data) {
        $scheme = new SchemeOfWork(
            $data['title'],
            $data['subject'],
            $data['grade_level'],
            $data['term'],
            $data['year'],
            $data['file_path'],
            $data['file_type']
        );
        
        $scheme->setId($data['id']);
        $scheme->setDescription($data['description']);
        $scheme->setFileSize($data['file_size']);
        $scheme->setActive($data['is_active']);
        
        return $scheme;
    }
}
?>