<?php
class PaymentApplicationService {
    private $mpesaGateway;
    private $transactionRepository;
    private $orderRepository;
    private $orderApplicationService;
    
    public function __construct() {
        $mpesaConfig = include '../config/mpesa.php';
        $this->mpesaGateway = new MpesaGateway($mpesaConfig);
        $this->transactionRepository = new MySQLTransactionRepository();
        $this->orderRepository = new MySQLOrderRepository();
        $this->orderApplicationService = new OrderApplicationService();
    }
    
    public function initiatePayment($phoneNumber, $cartData, $customerName = '') {
        try {
            // Create order
            $orderData = $this->orderApplicationService->createOrder($phoneNumber, $cartData, $customerName);
            
            // Create transaction
            $transactionId = new TransactionId();
            $transaction = new Transaction(
                $transactionId->getValue(),
                $orderData['order_id'],
                $orderData['total_amount'],
                $phoneNumber
            );
            
            $transaction = $this->transactionRepository->save($transaction);
            
            // Initiate M-Pesa payment
            $phoneNumberVO = new PhoneNumber($phoneNumber);
            $amountVO = new Money($orderData['total_amount']);
            
            $mpesaResponse = $this->mpesaGateway->initiateSTKPush(
                $phoneNumberVO,
                $amountVO,
                $transactionId,
                $orderData['order_number'],
                'Purchase of Teaching Schemes'
            );
            
            if ($mpesaResponse['ResponseCode'] == '0') {
                // Update transaction with checkout request ID
                $transaction->setCheckoutRequestId($mpesaResponse['CheckoutRequestID']);
                $this->transactionRepository->update($transaction);
                
                return [
                    'success' => true,
                    'transaction_id' => $transactionId->getValue(),
                    'order_id' => $orderData['order_id'],
                    'message' => 'Payment initiated successfully'
                ];
            } else {
                throw new Exception('Payment initiation failed: ' . $mpesaResponse['ResponseDescription']);
            }
            
        } catch (Exception $e) {
            throw new Exception('Payment service error: ' . $e->getMessage());
        }
    }
    
    public function handleCallback($callbackData) {
        try {
            $result = $this->mpesaGateway->handleCallback($callbackData);
            
            if ($result['success']) {
                // Find and update transaction
                $transaction = $this->transactionRepository->findByCheckoutRequestId($result['checkout_request_id']);
                
                if ($transaction) {
                    $transaction->markAsCompleted($result['mpesa_receipt_number']);
                    $this->transactionRepository->update($transaction);
                    
                    // Mark order as completed and create download tokens
                    $this->completeOrder($transaction->getOrderId());
                    
                    return ['success' => true, 'order_id' => $transaction->getOrderId()];
                }
            } else {
                // Handle failed payment
                $transaction = $this->transactionRepository->findByCheckoutRequestId($result['checkout_request_id']);
                if ($transaction) {
                    $transaction->markAsFailed();
                    $this->transactionRepository->update($transaction);
                }
            }
            
            return ['success' => $result['success']];
            
        } catch (Exception $e) {
            error_log('Payment callback error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    private function completeOrder($orderId) {
        $order = $this->orderRepository->findById($orderId);
        
        if ($order) {
            $order->markAsCompleted();
            $this->orderRepository->update($order);
            
            // Create download tokens
            $this->createDownloadTokens($orderId);
        }
    }
    
    private function createDownloadTokens($orderId) {
        $db = Database::getInstance()->getConnection();
        
        // Get order schemes
        $stmt = $db->prepare("
            SELECT s.id 
            FROM schemes s 
            JOIN order_items oi ON s.id = oi.scheme_id 
            WHERE oi.order_id = ?
        ");
        $stmt->execute([$orderId]);
        $schemes = $stmt->fetchAll();
        
        foreach ($schemes as $scheme) {
            $token = bin2hex(random_bytes(32));
            $expiresAt = date('Y-m-d H:i:s', strtotime('+30 days'));
            
            $stmt = $db->prepare("
                INSERT INTO downloads (order_id, scheme_id, download_token, expires_at) 
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                download_token = VALUES(download_token), 
                expires_at = VALUES(expires_at)
            ");
            $stmt->execute([$orderId, $scheme['id'], $token, $expiresAt]);
        }
    }
}
?>