<?php
class OrderApplicationService {
    private $orderRepository;
    private $customerRepository;
    private $schemeRepository;
    
    public function __construct() {
        $this->orderRepository = new MySQLOrderRepository();
        $this->customerRepository = new MySQLCustomerRepository();
        $this->schemeRepository = new MySQLSchemeRepository();
    }
    
    public function createOrder($phoneNumber, $cartData, $customerName = '') {
        try {
            // Validate phone number
            $phoneNumberVO = new PhoneNumber($phoneNumber);
            
            // Calculate total amount
            $totalAmount = PricingService::calculateTotal(count($cartData));
            
            // Get or create customer
            $customer = $this->customerRepository->findByPhoneNumber($phoneNumber);
            if (!$customer) {
                $nameParts = explode(' ', trim($customerName), 2);
                $firstName = $nameParts[0] ?? '';
                $lastName = $nameParts[1] ?? '';
                
                $customer = new Customer($phoneNumber, $firstName, $lastName);
                $customer = $this->customerRepository->save($customer);
            }
            
            // Generate order number
            $orderNumber = 'ORD-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Create order
            $order = new Order(
                $orderNumber, 
                $customer->getId(), 
                count($cartData), 
                $totalAmount->getAmount()
            );
            
            // Add order items
            foreach ($cartData as $item) {
                $schemeId = $this->mapItemIdToSchemeId($item['id']);
                $orderItem = new OrderItem($order->getId(), $schemeId);
                $order->addItem($orderItem);
            }
            
            // Save order
            $order = $this->orderRepository->save($order);
            
            return [
                'order_id' => $order->getId(),
                'order_number' => $order->getOrderNumber(),
                'total_amount' => $totalAmount->getAmount()
            ];
            
        } catch (Exception $e) {
            throw new Exception('Failed to create order: ' . $e->getMessage());
        }
    }
    
    private function mapItemIdToSchemeId($itemId) {
        // Simple mapping - in production, you'd have a proper mapping table
        $mapping = [
            'math-grade1-3-t1' => 1,
            'eng-grade4-6-t1' => 2,
            'sci-grade7-8-t1' => 3,
            'social-grade1-3-t2' => 4,
            'kisw-grade4-6-t2' => 5,
            'arts-grade1-6-t3' => 6
        ];
        
        return $mapping[$itemId] ?? 1;
    }
}
?>