<?php
class DownloadController {
    private $orderRepository;
    
    public function __construct() {
        $this->orderRepository = new MySQLOrderRepository();
    }
    
    public function downloadScheme($orderId, $schemeId) {
        try {
            // Verify download token
            $token = $_GET['token'] ?? '';
            if (!$this->verifyDownloadToken($orderId, $schemeId, $token)) {
                http_response_code(403);
                echo json_encode(['error' => 'Invalid or expired download token']);
                return;
            }
            
            // Get scheme file
            $scheme = $this->getSchemeById($schemeId);
            if (!$scheme || !file_exists($scheme['file_path'])) {
                http_response_code(404);
                echo json_encode(['error' => 'File not found']);
                return;
            }
            
            // Update download count
            $this->incrementDownloadCount($orderId, $schemeId);
            
            // Serve file
            $this->serveFile($scheme);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }
    
    public function getDownloadLinks($orderId) {
        try {
            $db = Database::getInstance()->getConnection();
            
            $stmt = $db->prepare("
                SELECT s.*, d.download_token, d.download_count, d.max_downloads, d.expires_at
                FROM schemes s
                JOIN order_items oi ON s.id = oi.scheme_id
                JOIN downloads d ON d.order_id = oi.order_id AND d.scheme_id = s.id
                WHERE oi.order_id = ? AND d.expires_at > NOW()
            ");
            $stmt->execute([$orderId]);
            $schemes = $stmt->fetchAll();
            
            $links = [];
            foreach ($schemes as $scheme) {
                $links[] = [
                    'scheme_id' => $scheme['id'],
                    'title' => $scheme['title'],
                    'file_type' => $scheme['file_type'],
                    'download_url' => "/download/scheme/{$orderId}/{$scheme['id']}?token={$scheme['download_token']}",
                    'downloads_remaining' => $scheme['max_downloads'] - $scheme['download_count']
                ];
            }
            
            header('Content-Type: application/json');
            echo json_encode(['links' => $links]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }
    
    private function verifyDownloadToken($orderId, $schemeId, $token) {
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            SELECT id, download_count, max_downloads, expires_at 
            FROM downloads 
            WHERE order_id = ? AND scheme_id = ? AND download_token = ? AND expires_at > NOW()
        ");
        $stmt->execute([$orderId, $schemeId, $token]);
        $download = $stmt->fetch();
        
        if (!$download) return false;
        
        // Check download limit
        if ($download['download_count'] >= $download['max_downloads']) {
            return false;
        }
        
        return true;
    }
    
    private function getSchemeById($schemeId) {
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("SELECT * FROM schemes WHERE id = ? AND is_active = 1");
        $stmt->execute([$schemeId]);
        return $stmt->fetch();
    }
    
    private function incrementDownloadCount($orderId, $schemeId) {
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            UPDATE downloads 
            SET download_count = download_count + 1 
            WHERE order_id = ? AND scheme_id = ?
        ");
        $stmt->execute([$orderId, $schemeId]);
    }
    
    private function serveFile($scheme) {
        $filePath = $scheme['file_path'];
        $fileName = $scheme['title'] . '.' . ($scheme['file_type'] === 'word' ? 'docx' : 'pdf');
        
        // Set headers for file download
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');
        header('Content-Length: ' . filesize($filePath));
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
        
        // Output file
        readfile($filePath);
        exit;
    }
}
?>