<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TeacherSchemes - Download Teaching Materials</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 50%, #1e3c72 100%);
            min-height: 100vh;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        /* Header */
        header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 1rem 0;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
            box-shadow: 0 2px 20px rgba(0, 0, 0, 0.1);
        }
        
        nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            font-size: 1.8rem;
            font-weight: bold;
            color: #e67e22;
        }
        
        .nav-links {
            display: flex;
            list-style: none;
            gap: 2rem;
            align-items: center;
        }
        
        .nav-links a {
            text-decoration: none;
            color: #333;
            font-weight: 500;
            transition: color 0.3s ease;
        }
        
        .nav-links a:hover {
            color: #e67e22;
        }
        
        /* Cart Button */
        .cart-btn {
            background: linear-gradient(135deg, #e67e22 0%, #d35400 100%);
            color: white;
            border: none;
            padding: 0.8rem 1.5rem;
            border-radius: 50px;
            font-weight: 600;
            cursor: pointer;
            position: relative;
            transition: all 0.3s ease;
        }
        
        .cart-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(230, 126, 34, 0.3);
        }
        
        .cart-count {
            position: absolute;
            top: -8px;
            right: -8px;
            background: #e74c3c;
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            font-size: 0.8rem;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        
        /* Main Content */
        main {
            margin-top: 100px;
            padding: 2rem 0;
        }
        
        .hero {
            text-align: center;
            padding: 4rem 0;
            color: white;
        }
        
        .hero h1 {
            font-size: 3rem;
            margin-bottom: 1rem;
            animation: fadeInUp 1s ease;
        }
        
        .hero p {
            font-size: 1.2rem;
            margin-bottom: 2rem;
            animation: fadeInUp 1s ease 0.2s both;
        }
        
        .pricing-info {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            padding: 1.5rem;
            border-radius: 15px;
            margin-top: 2rem;
            animation: fadeInUp 1s ease 0.4s both;
        }
        
        .pricing-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .pricing-item {
            text-align: center;
            padding: 1rem;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 10px;
        }
        
        .pricing-item .quantity {
            font-size: 1.2rem;
            font-weight: bold;
            color: #f39c12;
        }
        
        .pricing-item .price {
            font-size: 1.4rem;
            font-weight: bold;
            color: white;
        }
        
        /* Filters */
        .filters-section {
            background: white;
            padding: 2rem;
            border-radius: 20px;
            margin: 2rem 0;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .filters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .filter-group select {
            width: 100%;
            padding: 0.8rem;
            border: 2px solid #e1e1e1;
            border-radius: 10px;
            font-size: 1rem;
            background: white;
            transition: border-color 0.3s ease;
        }
        
        .filter-group select:focus {
            outline: none;
            border-color: #e67e22;
        }
        
        /* Scheme Cards */
        .schemes-section {
            background: white;
            margin: 2rem 0;
            padding: 3rem 0;
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .section-title {
            text-align: center;
            font-size: 2.5rem;
            color: #333;
            margin-bottom: 3rem;
        }
        
        .schemes-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 2rem;
            padding: 0 2rem;
        }
        
        .scheme-card {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border: 2px solid transparent;
            position: relative;
        }
        
        .scheme-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(0, 0, 0, 0.15);
            border-color: #e67e22;
        }
        
        .scheme-card.in-cart {
            border-color: #27ae60;
            background: linear-gradient(135deg, #f8fff8 0%, #ffffff 100%);
        }
        
        .scheme-card.in-cart::before {
            content: "✓";
            position: absolute;
            top: 15px;
            right: 15px;
            background: #27ae60;
            color: white;
            width: 25px;
            height: 25px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 0.9rem;
        }
        
        .scheme-card h3 {
            color: #2c3e50;
            margin-bottom: 1rem;
            font-size: 1.4rem;
        }
        
        .scheme-meta {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 0.5rem;
            margin-bottom: 1rem;
            font-size: 0.9rem;
            color: #666;
        }
        
        .scheme-meta span {
            padding: 0.3rem 0.8rem;
            background: #f8f9fa;
            border-radius: 15px;
            text-align: center;
        }
        
        .scheme-description {
            color: #555;
            margin-bottom: 1.5rem;
            font-size: 0.95rem;
        }
        
        .scheme-actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn-add-cart {
            background: linear-gradient(135deg, #27ae60 0%, #229954 100%);
            color: white;
            border: none;
            padding: 0.8rem 1.5rem;
            border-radius: 25px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            flex: 1;
        }
        
        .btn-add-cart:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(39, 174, 96, 0.3);
        }
        
        .btn-add-cart:disabled {
            background: #bdc3c7;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        .btn-remove-cart {
            background: linear-gradient(135deg, #e74c3c 0%, #c0392b 100%);
            color: white;
            border: none;
            padding: 0.8rem 1rem;
            border-radius: 25px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-remove-cart:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(231, 76, 60, 0.3);
        }
        
        /* Cart Sidebar */
        .cart-sidebar {
            position: fixed;
            top: 0;
            right: -400px;
            width: 400px;
            height: 100vh;
            background: white;
            box-shadow: -5px 0 20px rgba(0, 0, 0, 0.1);
            transition: right 0.3s ease;
            z-index: 2000;
            overflow-y: auto;
        }
        
        .cart-sidebar.open {
            right: 0;
        }
        
        .cart-header {
            padding: 2rem;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            background: white;
            z-index: 10;
        }
        
        .cart-header h3 {
            color: #2c3e50;
            font-size: 1.5rem;
        }
        
        .close-cart {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #666;
        }
        
        .cart-items {
            padding: 1rem 2rem;
        }
        
        .cart-item {
            padding: 1rem 0;
            border-bottom: 1px solid #f0f0f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .cart-item-info h4 {
            color: #2c3e50;
            font-size: 1rem;
            margin-bottom: 0.5rem;
        }
        
        .cart-item-meta {
            font-size: 0.8rem;
            color: #666;
        }
        
        .cart-total {
            padding: 2rem;
            border-top: 2px solid #e67e22;
            background: #f8f9fa;
            position: sticky;
            bottom: 0;
        }
        
        .total-breakdown {
            margin-bottom: 1rem;
        }
        
        .total-line {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
        }
        
        .total-final {
            font-size: 1.3rem;
            font-weight: bold;
            color: #e67e22;
            border-top: 1px solid #ddd;
            padding-top: 0.5rem;
        }
        
        .btn-checkout {
            background: linear-gradient(135deg, #e67e22 0%, #d35400 100%);
            color: white;
            border: none;
            padding: 1rem 2rem;
            border-radius: 50px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            width: 100%;
            margin-top: 1rem;
        }
        
        .btn-checkout:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(230, 126, 34, 0.3);
        }
        
        .btn-checkout:disabled {
            background: #bdc3c7;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        /* History Section */
        .history-section {
            background: white;
            margin: 2rem 0;
            padding: 3rem 0;
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .history-filters {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            padding: 0 2rem;
        }
        
        .history-item {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 1.5rem;
            margin-bottom: 1rem;
            border-left: 4px solid #e67e22;
        }
        
        .history-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }
        
        .order-number {
            font-weight: bold;
            color: #2c3e50;
        }
        
        .order-date {
            color: #666;
            font-size: 0.9rem;
        }
        
        .history-schemes {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .history-scheme {
            background: white;
            padding: 0.8rem;
            border-radius: 8px;
            font-size: 0.9rem;
        }
        
        .btn-redownload {
            background: #3498db;
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.9rem;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-redownload:hover {
            background: #2980b9;
            transform: translateY(-1px);
        }
        
        /* Payment Modal */
        .modal {
            display: none;
            position: fixed;
            z-index: 3000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(5px);
        }
        
        .modal-content {
            background-color: white;
            margin: 5% auto;
            padding: 2rem;
            border-radius: 20px;
            width: 90%;
            max-width: 500px;
            position: relative;
            animation: modalSlideIn 0.3s ease;
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .close {
            color: #aaa;
            float: right;
            font-size: 2rem;
            font-weight: bold;
            cursor: pointer;
            line-height: 1;
        }
        
        .close:hover {
            color: #000;
        }
        
        .payment-form {
            margin-top: 2rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input {
            width: 100%;
            padding: 1rem;
            border: 2px solid #e1e1e1;
            border-radius: 10px;
            font-size: 1rem;
            transition: border-color 0.3s ease;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #e67e22;
        }
        
        .mpesa-info {
            background: #f8f9fa;
            padding: 1rem;
            border-radius: 10px;
            margin-bottom: 1rem;
            border-left: 4px solid #e67e22;
        }
        
        .mpesa-steps {
            list-style: none;
            padding: 0;
            counter-reset: step-counter;
        }
        
        .mpesa-steps li {
            padding: 0.5rem 0;
            position: relative;
            padding-left: 2rem;
        }
        
        .mpesa-steps li:before {
            content: counter(step-counter);
            counter-increment: step-counter;
            position: absolute;
            left: 0;
            top: 0.5rem;
            background: #e67e22;
            color: white;
            width: 1.5rem;
            height: 1.5rem;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.8rem;
            font-weight: bold;
        }
        
        /* Loading Spinner */
        .loading {
            display: none;
            text-align: center;
            padding: 2rem;
        }
        
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #e67e22;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 2s linear infinite;
            margin: 0 auto 1rem;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Success/Error Messages */
        .alert {
            padding: 1rem;
            border-radius: 10px;
            margin-bottom: 1rem;
            font-weight: 500;
        }
        
        .alert-success {
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        
        .alert-error {
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        
        .alert-info {
            background-color: #cce7ff;
            border: 1px solid #99d3ff;
            color: #004085;
        }
        
        /* Animations */
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .fade-in {
            animation: fadeInUp 0.6s ease;
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .nav-links {
                gap: 1rem;
            }
            
            .nav-links a:not(.cart-btn) {
                display: none;
            }
            
            .hero h1 {
                font-size: 2rem;
            }
            
            .hero p {
                font-size: 1rem;
            }
            
            .pricing-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .schemes-grid {
                grid-template-columns: 1fr;
                padding: 0 1rem;
            }
            
            .cart-sidebar {
                width: 100%;
                right: -100%;
            }
            
            .modal-content {
                margin: 10% auto;
                width: 95%;
            }
            
            .filters-grid {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 480px) {
            .container {
                padding: 0 10px;
            }
            
            .hero {
                padding: 2rem 0;
            }
            
            .schemes-section, .history-section {
                margin: 1rem 0;
                padding: 2rem 0;
            }
            
            .section-title {
                font-size: 2rem;
            }
            
            .pricing-grid {
                grid-template-columns: 1fr;
            }
        }
        
        /* Empty States */
        .empty-state {
            text-align: center;
            padding: 3rem;
            color: #666;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: #bdc3c7;
        }
        
        /* Download Progress */
        .download-progress {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 1rem;
            margin-top: 1rem;
            display: none;
        }
        
        .progress-bar {
            background: #e1e1e1;
            border-radius: 10px;
            height: 20px;
            overflow: hidden;
            margin-bottom: 0.5rem;
        }
        
        .progress-fill {
            background: linear-gradient(135deg, #27ae60 0%, #229954 100%);
            height: 100%;
            width: 0%;
            transition: width 0.3s ease;
        }
        
        .progress-text {
            text-align: center;
            font-size: 0.9rem;
            color: #666;
        }
    </style>
</head>
<body>
    <header>
        <nav class="container">
            <div class="logo">📚 TeacherSchemes</div>
            <ul class="nav-links">
                <li><a href="#home">Home</a></li>
                <li><a href="#schemes">Schemes</a></li>
                <li><a href="#history">History</a></li>
                <li><a href="#contact">Contact</a></li>
                <li>
                    <button class="cart-btn" onclick="toggleCart()">
                        🛒 Cart
                        <span class="cart-count" id="cartCount">0</span>
                    </button>
                </li>
            </ul>
        </nav>
    </header>

    <main>
        <section class="hero" id="home">
            <div class="container">
                <h1>Download Quality Teaching Schemes</h1>
                <p>Professional schemes of work for teachers across all subjects and grade levels</p>
                
                <div class="pricing-info">
                    <h3>💰 Smart Bulk Pricing</h3>
                    <div class="pricing-grid">
                        <div class="pricing-item">
                            <div class="quantity">1 Scheme</div>
                            <div class="price">KSH 30</div>
                        </div>
                        <div class="pricing-item">
                            <div class="quantity">2 Schemes</div>
                            <div class="price">KSH 40</div>
                        </div>
                        <div class="pricing-item">
                            <div class="quantity">3+ Schemes</div>
                            <div class="price">KSH 50+</div>
                        </div>
                        <div class="pricing-item">
                            <div class="quantity">Best Deal</div>
                            <div class="price">Auto Calculated</div>
                        </div>
                    </div>
                    <p style="margin-top: 1rem; font-size: 0.9rem; opacity: 0.9;">
                        💡 The more you buy, the more you save! Our smart pricing automatically gives you the best deal.
                    </p>
                </div>
            </div>
        </section>

        <!-- Filters Section -->
        <section class="filters-section">
            <div class="container">
                <h3>🔍 Find Your Schemes</h3>
                <div class="filters-grid">
                    <div class="filter-group">
                        <select id="subjectFilter" onchange="filterSchemes()">
                            <option value="">All Subjects</option>
                            <option value="mathematics">Mathematics</option>
                            <option value="english">English</option>
                            <option value="science">Science</option>
                            <option value="social_studies">Social Studies</option>
                            <option value="kiswahili">Kiswahili</option>
                            <option value="creative_arts">Creative Arts</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <select id="gradeFilter" onchange="filterSchemes()">
                            <option value="">All Grades</option>
                            <option value="1-3">Grade 1-3</option>
                            <option value="4-6">Grade 4-6</option>
                            <option value="7-8">Grade 7-8</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <select id="termFilter" onchange="filterSchemes()">
                            <option value="">All Terms</option>
                            <option value="term1">Term 1</option>
                            <option value="term2">Term 2</option>
                            <option value="term3">Term 3</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <select id="yearFilter" onchange="filterSchemes()">
                            <option value="">All Years</option>
                            <option value="2024">2024</option>
                            <option value="2025">2025</option>
                        </select>
                    </div>
                </div>
            </div>
        </section>

        <section class="schemes-section" id="schemes">
            <div class="container">
                <h2 class="section-title">Available Schemes</h2>
                <div class="schemes-grid" id="schemesGrid">
                    <!-- Sample Scheme Cards -->
                    <div class="scheme-card fade-in" data-subject="mathematics" data-grade="1-3" data-term="term1" data-year="2024">
                        <h3>Mathematics Grade 1-3 Scheme</h3>
                        <div class="scheme-meta">
                            <span>📊 Mathematics</span>
                            <span>🎓 Grade 1-3</span>
                            <span>📅 Term 1</span>
                            <span>📄 PDF</span>
                        </div>
                        <p class="scheme-description">Comprehensive mathematics scheme covering numeracy, basic operations, and problem-solving skills for early learners.</p>
                        <div class="scheme-actions">
                            <button class="btn-add-cart" onclick="addToCart('math-grade1-3-t1', 'Mathematics Grade 1-3 Term 1', 'mathematics', '1-3', 'term1', 'pdf')">
                                Add to Cart
                            </button>
                        </div>
                    </div>

                    <div class="scheme-card fade-in" data-subject="english" data-grade="4-6" data-term="term1" data-year="2024">
                        <h3>English Grade 4-6 Scheme</h3>
                        <div class="scheme-meta">
                            <span>📖 English</span>
                            <span>🎓 Grade 4-6</span>
                            <span>📅 Term 1</span>
                            <span>📄 PDF</span>
                        </div>
                        <p class="scheme-description">Complete English language scheme including reading comprehension, writing skills, and grammar fundamentals.</p>
                        <div class="scheme-actions">
                            <button class="btn-add-cart" onclick="addToCart('eng-grade4-6-t1', 'English Grade 4-6 Term 1', 'english', '4-6', 'term1', 'pdf')">
                                Add to Cart
                            </button>
                        </div>
                    </div>

                    <div class="scheme-card fade-in" data-subject="science" data-grade="7-8" data-term="term1" data-year="2024">
                        <h3>Science Grade 7-8 Scheme</h3>
                        <div class="scheme-meta">
                            <span>🔬 Science</span>
                            <span>🎓 Grade 7-8</span>
                            <span>📅 Term 1</span>
                            <span>📄 PDF</span>
                        </div>
                        <p class="scheme-description">Integrated science scheme covering biology, chemistry, and physics concepts with practical activities.</p>
                        <div class="scheme-actions">
                            <button class="btn-add-cart" onclick="addToCart('sci-grade7-8-t1', 'Science Grade 7-8 Term 1', 'science', '7-8', 'term1', 'pdf')">
                                Add to Cart
                            </button>
                        </div>
                    </div>

                    <div class="scheme-card fade-in" data-subject="social_studies" data-grade="1-3" data-term="term2" data-year="2024">
                        <h3>Social Studies Grade 1-3 Scheme</h3>
                        <div class="scheme-meta">
                            <span>🌍 Social Studies</span>
                            <span>🎓 Grade 1-3</span>
                            <span>📅 Term 2</span>
                            <span>📄 Word</span>
                        </div>
                        <p class="scheme-description">Foundational social studies covering community, environment, and basic citizenship concepts.</p>
                        <div class="scheme-actions">
                            <button class="btn-add-cart" onclick="addToCart('social-grade1-3-t2', 'Social Studies Grade 1-3 Term 2', 'social_studies', '1-3', 'term2', 'word')">
                                Add to Cart
                            </button>
                        </div>
                    </div>

                    <div class="scheme-card fade-in" data-subject="kiswahili" data-grade="4-6" data-term="term2" data-year="2024">
                        <h3>Kiswahili Grade 4-6 Scheme</h3>
                        <div class="scheme-meta">
                            <span>🗣️ Kiswahili</span>
                            <span>🎓 Grade 4-6</span>
                            <span>📅 Term 2</span>
                            <span>📄 PDF</span>
                        </div>
                        <p class="scheme-description">Complete Kiswahili language scheme with focus on comprehension, composition, and oral skills.</p>
                        <div class="scheme-actions">
                            <button class="btn-add-cart" onclick="addToCart('kisw-grade4-6-t2', 'Kiswahili Grade 4-6 Term 2', 'kiswahili', '4-6', 'term2', 'pdf')">
                                Add to Cart
                            </button>
                        </div>
                    </div>

                    <div class="scheme-card fade-in" data-subject="creative_arts" data-grade="1-6" data-term="term3" data-year="2024">
                        <h3>Creative Arts Grade 1-6 Scheme</h3>
                        <div class="scheme-meta">
                            <span>🎨 Creative Arts</span>
                            <span>🎓 Grade 1-6</span>
                            <span>📅 Term 3</span>
                            <span>📄 PDF</span>
                        </div>
                        <p class="scheme-description">Comprehensive creative arts scheme including music, art, and craft activities for holistic development.</p>
                        <div class="scheme-actions">
                            <button class="btn-add-cart" onclick="addToCart('arts-grade1-6-t3', 'Creative Arts Grade 1-6 Term 3', 'creative_arts', '1-6', 'term3', 'pdf')">
                                Add to Cart
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </section>

        <!-- History Section -->
        <section class="history-section" id="history">
            <div class="container">
                <h2 class="section-title">📚 Purchase History</h2>
                
                <div class="history-filters">
                    <select id="historyTermFilter" onchange="filterHistory()">
                        <option value="">All Terms</option>
                        <option value="term1">Term 1</option>
                        <option value="term2">Term 2</option>
                        <option value="term3">Term 3</option>
                    </select>
                    <select id="historyYearFilter" onchange="filterHistory()">
                        <option value="">All Years</option>
                        <option value="2024">2024</option>
                        <option value="2025">2025</option>
                    </select>
                </div>

                <div id="historyContainer">
                    <!-- Sample History Items -->
                    <div class="history-item" data-term="term1" data-year="2024">
                        <div class="history-header">
                            <span class="order-number">Order #ORD-20240115-001</span>
                            <span class="order-date">January 15, 2024</span>
                        </div>
                        <div class="history-schemes">
                            <div class="history-scheme">Mathematics Grade 1-3 Term 1</div>
                            <div class="history-scheme">English Grade 4-6 Term 1</div>
                            <div class="history-scheme">Science Grade 7-8 Term 1</div>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <strong>Total: KSH 50 (3 items)</strong>
                            <button class="btn-redownload" onclick="redownloadOrder('ORD-20240115-001')">
                                ⬇️ Re-download All
                            </button>
                        </div>
                    </div>

                    <div class="history-item" data-term="term2" data-year="2024">
                        <div class="history-header">
                            <span class="order-number">Order #ORD-20240420-002</span>
                            <span class="order-date">April 20, 2024</span>
                        </div>
                        <div class="history-schemes">
                            <div class="history-scheme">Social Studies Grade 1-3 Term 2</div>
                            <div class="history-scheme">Kiswahili Grade 4-6 Term 2</div>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <strong>Total: KSH 40 (2 items)</strong>
                            <button class="btn-redownload" onclick="redownloadOrder('ORD-20240420-002')">
                                ⬇️ Re-download All
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </section>
    </main>

    <!-- Cart Sidebar -->
    <div class="cart-sidebar" id="cartSidebar">
        <div class="cart-header">
            <h3>🛒 Shopping Cart</h3>
            <button class="close-cart" onclick="toggleCart()">&times;</button>
        </div>
        
        <div class="cart-items" id="cartItems">
            <div class="empty-state">
                <div style="font-size: 3rem; margin-bottom: 1rem;">🛒</div>
                <p>Your cart is empty</p>
                <p style="font-size: 0.9rem; margin-top: 0.5rem;">Add schemes to get started!</p>
            </div>
        </div>
        
        <div class="cart-total" id="cartTotal" style="display: none;">
            <div class="total-breakdown" id="totalBreakdown"></div>
            <div class="total-line total-final">
                <span>Total:</span>
                <span id="finalTotal">KSH 0</span>
            </div>
            <button class="btn-checkout" onclick="proceedToCheckout()">
                Proceed to Payment
            </button>
        </div>
    </div>

    <!-- Payment Modal -->
    <div id="paymentModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closePaymentModal()">&times;</span>
            <h2>Complete Your Purchase</h2>
            
            <div id="orderSummary">
                <!-- Order summary will be populated here -->
            </div>

            <form class="payment-form" id="paymentForm" onsubmit="processPayment(event)">
                <div class="mpesa-info">
                    <h4>💳 M-Pesa Payment Process</h4>
                    <ol class="mpesa-steps">
                        <li>Enter your M-Pesa phone number below</li>
                        <li>Click "Pay with M-Pesa" button</li>
                        <li>Check your phone for M-Pesa prompt</li>
                        <li>Enter your M-Pesa PIN to complete payment</li>
                        <li>Download your schemes immediately</li>
                    </ol>
                </div>

                <div class="form-group">
                    <label for="phoneNumber">M-Pesa Phone Number</label>
                    <input type="tel" id="phoneNumber" name="phone_number" 
                           placeholder="254712345678" required 
                           pattern="254[0-9]{9}" 
                           title="Please enter a valid Kenyan phone number starting with 254">
                </div>

                <div class="form-group">
                    <label for="customerName">Your Name (Optional)</label>
                    <input type="text" id="customerName" name="customer_name" 
                           placeholder="Enter your full name">
                </div>

                <input type="hidden" id="cartData" name="cart_data">
                <input type="hidden" id="totalAmount" name="total_amount">

                <button type="submit" class="btn-checkout">
                    Pay with M-Pesa
                </button>
            </form>

            <div class="loading" id="loadingDiv">
                <div class="spinner"></div>
                <p>Processing your payment... Please check your phone for M-Pesa prompt.</p>
            </div>

            <div class="download-progress" id="downloadProgress">
                <div class="progress-bar">
                    <div class="progress-fill" id="progressFill"></div>
                </div>
                <div class="progress-text" id="progressText">Preparing downloads...</div>
            </div>
        </div>
    </div>

    <script>
        // Cart Management
        let cart = [];
        
        function addToCart(id, name, subject, grade, term, format) {
            // Check if item already in cart
            if (cart.find(item => item.id === id)) {
                alert('This scheme is already in your cart!');
                return;
            }
            
            cart.push({ id, name, subject, grade, term, format });
            updateCartUI();
            updateSchemeCards();
            
            // Show feedback
            const button = event.target;
            const originalText = button.textContent;
            button.textContent = '✓ Added!';
            button.style.background = '#27ae60';
            setTimeout(() => {
                updateSchemeCards(); // This will restore the button state
            }, 1000);
        }
        
        function removeFromCart(id) {
            cart = cart.filter(item => item.id !== id);
            updateCartUI();
            updateSchemeCards();
        }
        
        function updateCartUI() {
            const cartCount = document.getElementById('cartCount');
            const cartItems = document.getElementById('cartItems');
            const cartTotal = document.getElementById('cartTotal');
            
            cartCount.textContent = cart.length;
            cartCount.style.display = cart.length > 0 ? 'flex' : 'none';
            
            if (cart.length === 0) {
                cartItems.innerHTML = `
                    <div class="empty-state">
                        <div style="font-size: 3rem; margin-bottom: 1rem;">🛒</div>
                        <p>Your cart is empty</p>
                        <p style="font-size: 0.9rem; margin-top: 0.5rem;">Add schemes to get started!</p>
                    </div>
                `;
                cartTotal.style.display = 'none';
                return;
            }
            
            cartItems.innerHTML = cart.map(item => `
                <div class="cart-item">
                    <div class="cart-item-info">
                        <h4>${item.name}</h4>
                        <div class="cart-item-meta">${item.subject} • ${item.grade} • ${item.format.toUpperCase()}</div>
                    </div>
                    <button class="btn-remove-cart" onclick="removeFromCart('${item.id}')">×</button>
                </div>
            `).join('');
            
            updateCartTotal();
            cartTotal.style.display = 'block';
        }
        
        function updateCartTotal() {
            const count = cart.length;
            const total = calculateTotal(count);
            const breakdown = getBreakdown(count);
            
            document.getElementById('totalBreakdown').innerHTML = breakdown;
            document.getElementById('finalTotal').textContent = `KSH ${total}`;
        }
        
        function calculateTotal(count) {
            if (count <= 0) return 0;
            if (count === 1) return 30;
            if (count === 2) return 40;
            
            // For 3+ items, use the most economical combination
            const groupsOfThree = Math.floor(count / 3);
            const remainder = count % 3;
            
            let total = groupsOfThree * 50;
            
            if (remainder === 1) total += 30;
            if (remainder === 2) total += 40;
            
            return total;
        }
        
        function getBreakdown(count) {
            if (count <= 0) return '';
            if (count === 1) return '<div class="total-line"><span>1 scheme:</span><span>KSH 30</span></div>';
            if (count === 2) return '<div class="total-line"><span>2 schemes:</span><span>KSH 40</span></div>';
            
            const groupsOfThree = Math.floor(count / 3);
            const remainder = count % 3;
            let breakdown = '';
            
            if (groupsOfThree > 0) {
                breakdown += `<div class="total-line"><span>${groupsOfThree} group(s) of 3:</span><span>KSH ${groupsOfThree * 50}</span></div>`;
            }
            
            if (remainder === 1) {
                breakdown += '<div class="total-line"><span>1 additional:</span><span>KSH 30</span></div>';
            } else if (remainder === 2) {
                breakdown += '<div class="total-line"><span>2 additional:</span><span>KSH 40</span></div>';
            }
            
            return breakdown;
        }
        
        function updateSchemeCards() {
            document.querySelectorAll('.scheme-card').forEach(card => {
                const button = card.querySelector('.btn-add-cart');
                const schemeId = button.getAttribute('onclick').match(/'([^']+)'/)[1];
                
                const inCart = cart.find(item => item.id === schemeId);
                
                if (inCart) {
                    card.classList.add('in-cart');
                    button.textContent = 'In Cart';
                    button.disabled = true;
                    
                    // Add remove button if not exists
                    if (!card.querySelector('.btn-remove-cart')) {
                        const removeBtn = document.createElement('button');
                        removeBtn.className = 'btn-remove-cart';
                        removeBtn.textContent = 'Remove';
                        removeBtn.onclick = () => removeFromCart(schemeId);
                        card.querySelector('.scheme-actions').appendChild(removeBtn);
                    }
                } else {
                    card.classList.remove('in-cart');
                    button.textContent = 'Add to Cart';
                    button.disabled = false;
                    
                    // Remove the remove button
                    const removeBtn = card.querySelector('.btn-remove-cart');
                    if (removeBtn) removeBtn.remove();
                }
            });
        }
        
        function toggleCart() {
            const sidebar = document.getElementById('cartSidebar');
            sidebar.classList.toggle('open');
        }
        
        function proceedToCheckout() {
            if (cart.length === 0) return;
            
            const total = calculateTotal(cart.length);
            
            document.getElementById('orderSummary').innerHTML = `
                <div class="alert alert-info">
                    <h4>Order Summary</h4>
                    <p><strong>${cart.length} scheme(s)</strong> - Total: <strong>KSH ${total}</strong></p>
                    <div style="margin-top: 1rem; font-size: 0.9rem;">
                        ${cart.map(item => `• ${item.name}`).join('<br>')}
                    </div>
                </div>
            `;
            
            document.getElementById('cartData').value = JSON.stringify(cart);
            document.getElementById('totalAmount').value = total;
            document.getElementById('paymentModal').style.display = 'block';
        }
        
        function closePaymentModal() {
            document.getElementById('paymentModal').style.display = 'none';
            document.getElementById('loadingDiv').style.display = 'none';
            document.getElementById('paymentForm').style.display = 'block';
            document.getElementById('downloadProgress').style.display = 'none';
        }
        
        function processPayment(event) {
            event.preventDefault();
            
            const formData = new FormData(event.target);
            const phoneNumber = formData.get('phone_number');
            
            // Validate phone number format
            const phoneRegex = /^254[0-9]{9}$/;
            if (!phoneRegex.test(phoneNumber)) {
                alert('Please enter a valid Kenyan phone number starting with 254');
                return;
            }

            // Show loading state
            document.getElementById('paymentForm').style.display = 'none';
            document.getElementById('loadingDiv').style.display = 'block';

            // Send payment request to backend
            fetch('/payment/initiate', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Poll for payment status
                    checkPaymentStatus(data.transaction_id);
                } else {
                    throw new Error(data.message || 'Payment initiation failed');
                }
            })
            .catch(error => {
                alert('Error: ' + error.message);
                closePaymentModal();
            });
        }
        
        function checkPaymentStatus(transactionId) {
            const checkStatus = () => {
                fetch(`/payment/status/${transactionId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'completed') {
                        // Start downloads immediately
                        startDownloads(data.order_id);
                    } else if (data.status === 'failed') {
                        alert('Payment failed. Please try again.');
                        closePaymentModal();
                    } else {
                        // Still pending, check again in 3 seconds
                        setTimeout(checkStatus, 3000);
                    }
                })
                .catch(error => {
                    console.error('Status check error:', error);
                    setTimeout(checkStatus, 5000); // Retry after 5 seconds
                });
            };

            // Start checking status
            setTimeout(checkStatus, 3000);
        }
        
        function startDownloads(orderId) {
            document.getElementById('loadingDiv').style.display = 'none';
            document.getElementById('downloadProgress').style.display = 'block';
            
            const progressFill = document.getElementById('progressFill');
            const progressText = document.getElementById('progressText');
            
            // Simulate download progress for each item
            let downloaded = 0;
            const total = cart.length;
            
            cart.forEach((item, index) => {
                setTimeout(() => {
                    downloadScheme(orderId, item.id, item.name).then(() => {
                        downloaded++;
                        const progress = (downloaded / total) * 100;
                        progressFill.style.width = progress + '%';
                        progressText.textContent = `Downloaded ${downloaded} of ${total} schemes`;
                        
                        if (downloaded === total) {
                            setTimeout(() => {
                                progressText.textContent = 'All downloads complete! ✅';
                                setTimeout(() => {
                                    closePaymentModal();
                                    cart = []; // Clear cart
                                    updateCartUI();
                                    updateSchemeCards();
                                    toggleCart(); // Close cart
                                    alert('All schemes downloaded successfully!');
                                }, 2000);
                            }, 500);
                        }
                    });
                }, index * 1000); // Stagger downloads
            });
        }
        
        function downloadScheme(orderId, schemeId, schemeName) {
            return new Promise((resolve) => {
                // Simulate download API call
                fetch(`/download/scheme/${orderId}/${schemeId}`)
                .then(response => response.blob())
                .then(blob => {
                    // Create download link
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.style.display = 'none';
                    a.href = url;
                    a.download = `${schemeName}.pdf`;
                    document.body.appendChild(a);
                    a.click();
                    window.URL.revokeObjectURL(url);
                    document.body.removeChild(a);
                    resolve();
                })
                .catch(error => {
                    console.error('Download error:', error);
                    resolve(); // Continue with other downloads
                });
            });
        }
        
        function redownloadOrder(orderNumber) {
            // Simulate re-download
            const progressBar = event.target;
            const originalText = progressBar.textContent;
            progressBar.textContent = '⬇️ Downloading...';
            progressBar.disabled = true;
            
            setTimeout(() => {
                progressBar.textContent = '✅ Downloaded!';
                setTimeout(() => {
                    progressBar.textContent = originalText;
                    progressBar.disabled = false;
                }, 2000);
            }, 2000);
        }
        
        // Filtering Functions
        function filterSchemes() {
            const subject = document.getElementById('subjectFilter').value;
            const grade = document.getElementById('gradeFilter').value;
            const term = document.getElementById('termFilter').value;
            const year = document.getElementById('yearFilter').value;
            
            document.querySelectorAll('.scheme-card').forEach(card => {
                const cardSubject = card.getAttribute('data-subject');
                const cardGrade = card.getAttribute('data-grade');
                const cardTerm = card.getAttribute('data-term');
                const cardYear = card.getAttribute('data-year');
                
                const subjectMatch = !subject || cardSubject === subject;
                const gradeMatch = !grade || cardGrade === grade;
                const termMatch = !term || cardTerm === term;
                const yearMatch = !year || cardYear === year;
                
                if (subjectMatch && gradeMatch && termMatch && yearMatch) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });
        }
        
        function filterHistory() {
            const term = document.getElementById('historyTermFilter').value;
            const year = document.getElementById('historyYearFilter').value;
            
            document.querySelectorAll('.history-item').forEach(item => {
                const itemTerm = item.getAttribute('data-term');
                const itemYear = item.getAttribute('data-year');
                
                const termMatch = !term || itemTerm === term;
                const yearMatch = !year || itemYear === year;
                
                if (termMatch && yearMatch) {
                    item.style.display = 'block';
                } else {
                    item.style.display = 'none';
                }
            });
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('paymentModal');
            if (event.target === modal) {
                closePaymentModal();
            }
            
            const cartSidebar = document.getElementById('cartSidebar');
            if (event.target === cartSidebar) {
                toggleCart();
            }
        };

        // Format phone number as user types
        document.getElementById('phoneNumber').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, ''); // Remove non-digits
            
            // Ensure it starts with 254
            if (!value.startsWith('254') && value.length > 0) {
                if (value.startsWith('0')) {
                    value = '254' + value.substring(1);
                } else if (value.startsWith('7') || value.startsWith('1')) {
                    value = '254' + value;
                }
            }
            
            // Limit to 12 digits (254 + 9 digits)
            if (value.length > 12) {
                value = value.substring(0, 12);
            }
            
            e.target.value = value;
        });

        // Add animation to cards on scroll
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.style.animationDelay = '0.1s';
                    entry.target.classList.add('fade-in');
                }
            });
        }, observerOptions);

        // Observe all scheme cards
        document.querySelectorAll('.scheme-card').forEach(card => {
            observer.observe(card);
        });

        // Smooth scrolling for navigation links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            updateCartUI();
            updateSchemeCards();
        });
    </script>
</body>
</html>